"""
Blender UI Panel for Mocai Add-on
Displays controls for video upload and processing
"""

import bpy
from bpy.types import Panel


class MOCAI_PT_main_panel(Panel):
    """Main panel for Mocai add-on"""
    bl_label = "Mocai Motion Capture"
    bl_idname = "MOCAI_PT_main_panel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'Mocai'

    def draw(self, context):
        layout = self.layout
        props = context.scene.mocai

        # API Key Section
        box = layout.box()
        box.label(text="API Configuration", icon='USER')
        box.prop(props, "api_key")

        # Show authentication status
        row = box.row()
        if props.is_authenticated:
            row.label(text="✓ Authenticated", icon='CHECKMARK')
            row.operator("mocai.authenticate", text="Re-authenticate", icon='FILE_REFRESH')
        else:
            row.operator("mocai.authenticate", text="Authenticate", icon='LINKED')

        # Video Upload Section
        box = layout.box()
        box.label(text="Video Upload", icon='FILE_MOVIE')

        # Video file selection
        box.prop(props, "video_path")
        box.prop(props, "video_title")

        # Processing Options
        box.label(text="Processing Options:", icon='SETTINGS')
        box.prop(props, "character")

        # Export Format Options
        box.label(text="Export Formats:", icon='EXPORT')
        box.prop(props, "export_glb")

        # FBX with conditional availability
        row = box.row()
        row.enabled = props.character != 'Manny'
        row.prop(props, "export_fbx")
        if props.character == 'Manny':
            box.label(text="FBX not available for Manny character", icon='INFO')

        box.prop(props, "export_bvh")
        box.prop(props, "export_usd")

        # Motion Capture Options
        box.label(text="Motion Capture:", icon='ANIM')
        box.prop(props, "hand_motion_capture")
        box.prop(props, "depth")

        # Upload button
        row = box.row()
        row.scale_y = 1.5
        row.enabled = not props.is_processing
        row.operator("mocai.upload_video", icon='EXPORT')

        # Status Section
        box = layout.box()
        box.label(text="Processing Status", icon='INFO')

        if props.video_id:
            box.label(text=f"Video ID: {props.video_id}")

        if props.processing_status:
            # Status indicator with color
            status = props.processing_status
            if status == "processed":
                icon = 'CHECKMARK'
            elif status == "failed":
                icon = 'ERROR'
            elif status in ["queued", "processing"]:
                icon = 'SORTTIME'
            else:
                icon = 'QUESTION'

            box.label(text=f"Status: {status}", icon=icon)

        if props.status_message:
            # Wrap long messages
            for line in props.status_message.split('\n'):
                box.label(text=line)

        # Action buttons
        if props.is_processing:
            row = box.row()
            row.operator("mocai.cancel_processing", icon='CANCEL')
        elif props.video_id and props.processing_status == "processed":
            row = box.row()
            row.operator("mocai.download_assets", icon='IMPORT')

        # Advanced Settings Section
        box = layout.box()
        box.label(text="Advanced Settings", icon='PREFERENCES')
        box.prop(props, "poll_interval", text="Status Check Interval (seconds)")


class MOCAI_PT_help_panel(Panel):
    """Help panel for Mocai add-on"""
    bl_label = "Help & Info"
    bl_idname = "MOCAI_PT_help_panel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'Mocai'
    bl_options = {'DEFAULT_CLOSED'}

    def draw(self, context):
        layout = self.layout

        box = layout.box()
        box.label(text="Getting Started:", icon='INFO')
        box.label(text="1. Enter your Mocai API key")
        box.label(text="2. Click 'Authenticate' to verify")
        box.label(text="3. Select a video file to upload")
        box.label(text="4. Configure processing options")
        box.label(text="5. Click 'Upload Video'")
        box.label(text="6. Wait for processing to complete")
        box.label(text="7. Files will auto-import to scene")

        box = layout.box()
        box.label(text="Supported Characters:", icon='ARMATURE_DATA')
        box.label(text="• Manny (default)")
        box.label(text="• Male Bot / Female Bot")
        box.label(text="• Adam / James")
        box.label(text="• Megan / Sophie")


# Registration
classes = (
    MOCAI_PT_main_panel,
    MOCAI_PT_help_panel,
)


def register():
    """Register UI panels"""
    for cls in classes:
        bpy.utils.register_class(cls)


def unregister():
    """Unregister UI panels"""
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)
