"""
Blender Properties for Mocai Add-on
Stores user settings and processing state
"""

import bpy
from bpy.props import (
    StringProperty,
    BoolProperty,
    EnumProperty,
    IntProperty,
    PointerProperty
)


class MocaiProperties(bpy.types.PropertyGroup):
    """Property group for Mocai add-on settings and state"""

    def update_character(self, context):
        """Called when character selection changes"""
        # Disable FBX export for Manny character
        if self.character == 'Manny' and self.export_fbx:
            self.export_fbx = False

    # API Settings
    api_key: StringProperty(
        name="API Key",
        description="Your Mocai API key (starts with moc_test_sk_ or moc_prod_sk_)",
        default="",
        subtype='PASSWORD',
        update=lambda self, context: setattr(self, 'is_authenticated', False)
    )

    # Video Selection
    video_path: StringProperty(
        name="Video File",
        description="Path to the video file to upload",
        default="",
        subtype='FILE_PATH'
    )

    video_title: StringProperty(
        name="Title",
        description="Title for the video",
        default="Blender Upload"
    )

    # Processing Options
    character: EnumProperty(
        name="Character",
        description="Character model to use for animation",
        items=[
            ('Manny', 'Manny', 'Manny character'),
            ('Male Bot', 'Male Bot', 'Male Bot character'),
            ('Female Bot', 'Female Bot', 'Female Bot character'),
            ('Adam', 'Adam', 'Adam character'),
            ('James', 'James', 'James character'),
            ('Megan', 'Megan', 'Megan character'),
            ('Sophie', 'Sophie', 'Sophie character'),
        ],
        default='Manny',
        update=lambda self, context: self.update_character(context)
    )

    export_glb: BoolProperty(
        name="Export GLB",
        description="Export GLB format for direct import into Blender",
        default=True
    )

    export_fbx: BoolProperty(
        name="Export FBX",
        description="Export FBX format",
        default=False
    )

    export_bvh: BoolProperty(
        name="Export BVH",
        description="Export BVH format for motion data",
        default=False
    )

    export_usd: BoolProperty(
        name="Export USDZ",
        description="Export USDZ format (Packaged USD with textures)",
        default=False
    )

    hand_motion_capture: BoolProperty(
        name="Hand Motion Capture",
        description="Include hand motion capture",
        default=False
    )

    depth: BoolProperty(
        name="Depth Data",
        description="Include depth data for better 3D positioning",
        default=False
    )

    # Authentication State
    is_authenticated: BoolProperty(
        name="Is Authenticated",
        description="Whether the user is currently authenticated",
        default=False
    )

    # Processing State
    video_id: StringProperty(
        name="Video ID",
        description="ID of the currently processing video",
        default=""
    )

    processing_status: StringProperty(
        name="Status",
        description="Current processing status",
        default="idle"
    )

    status_message: StringProperty(
        name="Status Message",
        description="Status message for user feedback",
        default=""
    )

    is_processing: BoolProperty(
        name="Is Processing",
        description="Whether a video is currently being processed",
        default=False
    )

    poll_interval: IntProperty(
        name="Poll Interval",
        description="Seconds between status checks",
        default=5,
        min=2,
        max=30
    )

    # Download path
    download_path: StringProperty(
        name="Download Path",
        description="Path where GLB file will be downloaded",
        default=""
    )


def register():
    """Register properties"""
    bpy.utils.register_class(MocaiProperties)
    bpy.types.Scene.mocai = PointerProperty(type=MocaiProperties)


def unregister():
    """Unregister properties"""
    del bpy.types.Scene.mocai
    bpy.utils.unregister_class(MocaiProperties)
