"""
Mocai API Client for Blender Add-on
Handles authentication, video upload, status polling, and asset downloads
"""

import requests
import json
from typing import Optional, Dict, Any


class MocaiAPIClient:
    """Client for interacting with the Mocai API"""

    BASE_URL = "https://api.mocai.ai"

    def __init__(self, api_key: str):
        """
        Initialize the Mocai API client

        Args:
            api_key: Mocai API key (moc_test_sk_... or moc_prod_sk_...)
        """
        self.api_key = api_key
        self.jwt_token: Optional[str] = None
        self.session = requests.Session()

    def authenticate(self) -> bool:
        """
        Exchange API key for JWT token

        Returns:
            True if authentication successful, False otherwise
        """
        try:
            url = f"{self.BASE_URL}/api/v1/auth/token"
            headers = {
                "X-API-Key": self.api_key
            }

            # Authenticating with API
            response = self.session.post(url, headers=headers, timeout=10)

            if response.status_code != 200:
                # Authentication failed
                return False

            response.raise_for_status()

            data = response.json()
            self.jwt_token = data.get("access_token")

            if self.jwt_token:
                # Set authorization header for all future requests
                self.session.headers.update({
                    "Authorization": f"Bearer {self.jwt_token}"
                })
                # Authentication successful
                return True

            return False

        except requests.exceptions.Timeout:
            # Handle timeout
            return False
        except requests.exceptions.ConnectionError:
            # Handle connection error
            return False
        except requests.exceptions.RequestException as e:
            # Handle other network errors
            return False

    def upload_video(
        self,
        video_path: str,
        title: str = "Blender Upload",
        character: str = "Manny",
        export_glb: bool = True,
        export_fbx: bool = False,
        export_bvh: bool = False,
        export_usd: bool = False,
        hand_motion_capture: bool = False,
        depth: bool = False
    ) -> Optional[Dict[str, Any]]:
        """
        Upload a video file to Mocai for processing

        Args:
            video_path: Path to the video file
            title: Title for the video
            character: Character to use (Male Bot, Female Bot, Adam, James, Megan, Sophie, Manny)
            export_glb: Export GLB format
            export_fbx: Export FBX format
            export_bvh: Export BVH format
            export_usd: Export USD format
            hand_motion_capture: Include hand motion capture
            depth: Include depth data

        Returns:
            Dictionary with videoId and status, or None if failed
        """
        if not self.jwt_token:
            # Not authenticated
            return None

        try:
            import os
            url = f"{self.BASE_URL}/api/v1/videos/upload"

            # Get filename and prepare file with proper content type
            filename = os.path.basename(video_path)
            file_ext = os.path.splitext(filename)[1].lower()

            # Map file extensions to MIME types
            mime_types = {
                '.mp4': 'video/mp4',
                '.mov': 'video/quicktime',
                '.avi': 'video/x-msvideo',
                '.webm': 'video/webm'
            }

            content_type = mime_types.get(file_ext, 'video/mp4')

            # Prepare form data - include filename and content type
            files = {
                'video': (filename, open(video_path, 'rb'), content_type)
            }

            data = {
                'title': title,
                'character': character,
                'exportGLB': str(export_glb).lower(),
                'exportFBX': str(export_fbx).lower(),
                'exportBVH': str(export_bvh).lower(),
                'exportUSD': str(export_usd).lower(),
                'handMotionCapture': str(hand_motion_capture).lower(),
                'depth': str(depth).lower(),
                'jointAngles2D': 'false',
                'exportMasks': 'false'
            }

            # Upload video with form data

            response = self.session.post(url, files=files, data=data, timeout=300)

            # Check for errors and capture detailed error message
            if response.status_code != 200 and response.status_code != 201:
                try:
                    error_data = response.json()
                    error_msg = error_data.get('message', error_data.get('error', 'Unknown error'))
                    # Video upload failed with error message
                except:
                    # Video upload failed
                    pass
                return None

            response.raise_for_status()
            return response.json()

        except requests.exceptions.Timeout:
            # Upload timed out
            return None
        except requests.exceptions.ConnectionError:
            # Connection error during upload
            return None
        except requests.exceptions.RequestException as e:
            # Handle other upload errors
            return None
        finally:
            # Close the file handle (it's the second element in the tuple)
            if 'video' in files and len(files['video']) > 1:
                files['video'][1].close()

    def get_video_status(self, video_id: str) -> Optional[Dict[str, Any]]:
        """
        Get the processing status of a video

        Args:
            video_id: The video ID from upload response

        Returns:
            Dictionary with status and asset URLs, or None if failed
        """
        if not self.jwt_token:
            # Not authenticated
            return None

        try:
            url = f"{self.BASE_URL}/api/v1/videos/{video_id}/status"
            response = self.session.get(url, timeout=10)
            response.raise_for_status()

            return response.json()

        except requests.exceptions.Timeout:
            # Request timed out
            return None
        except requests.exceptions.ConnectionError:
            # Connection error
            return None
        except requests.exceptions.RequestException as e:
            # Failed to get video status
            return None

    def download_asset(self, url: str, output_path: str) -> bool:
        """
        Download an asset from a signed URL

        Args:
            url: Signed URL from video status response
            output_path: Path to save the downloaded file

        Returns:
            True if download successful, False otherwise
        """
        try:
            # Don't use session auth for signed URLs
            response = requests.get(url, stream=True, timeout=300)
            response.raise_for_status()

            with open(output_path, 'wb') as f:
                for chunk in response.iter_content(chunk_size=8192):
                    f.write(chunk)

            return True

        except requests.exceptions.Timeout:
            # Download timed out
            return False
        except requests.exceptions.ConnectionError:
            # Connection error during download
            return False
        except requests.exceptions.RequestException as e:
            # Failed to download asset
            return False

    def list_videos(
        self,
        page: int = 1,
        limit: int = 20,
        status: str = "all"
    ) -> Optional[Dict[str, Any]]:
        """
        List videos with pagination

        Args:
            page: Page number
            limit: Items per page
            status: Filter by status (all, queued, processing, completed, failed)

        Returns:
            Dictionary with videos array and pagination info, or None if failed
        """
        if not self.jwt_token:
            # Not authenticated
            return None

        try:
            url = f"{self.BASE_URL}/api/v1/videos"
            params = {
                "page": page,
                "limit": limit,
                "status": status,
                "sortBy": "createdAt",
                "sortOrder": "desc"
            }

            response = self.session.get(url, params=params, timeout=10)
            response.raise_for_status()

            return response.json()

        except requests.exceptions.Timeout:
            # Request timed out
            return None
        except requests.exceptions.ConnectionError:
            # Connection error
            return None
        except requests.exceptions.RequestException as e:
            # Failed to list videos
            return None
