"""
Mocai Blender Add-on
Upload videos to Mocai API and automatically import processed animations
"""

bl_info = {
    "name": "Mocai Motion Capture",
    "author": "Mocai Inc.",
    "version": (1, 0, 3),
    "blender": (3, 0, 0),
    "location": "View3D > Sidebar > Mocai",
    "description": "Upload videos to Mocai for motion capture processing and import animations",
    "warning": "",
    "doc_url": "https://mocai.ai",
    "category": "Animation",
}


# Auto-install dependencies
import sys
import subprocess
from importlib import util as importlib_util

def check_module(module_name):
    """Check if a module is installed"""
    return importlib_util.find_spec(module_name) is not None

def install_module(module_name):
    """Attempt to install a Python module using pip"""
    try:
        print(f"Mocai: Installing {module_name} module...")
        python_exe = sys.executable
        subprocess.check_call([python_exe, "-m", "pip", "install", module_name])
        print(f"Mocai: Successfully installed {module_name}")
        return True
    except subprocess.CalledProcessError as e:
        print(f"Mocai: Failed to install {module_name}: {e}")
        return False
    except Exception as e:
        print(f"Mocai: Error installing {module_name}: {e}")
        return False

def ensure_pip():
    """Ensure pip is available"""
    try:
        import pip
        return True
    except ImportError:
        print("Mocai: pip not found, attempting to install...")
        try:
            subprocess.check_call([sys.executable, "-m", "ensurepip", "--default-pip"])
            print("Mocai: pip installed successfully")
            return True
        except Exception as e:
            print(f"Mocai: Failed to install pip: {e}")
            return False

# Check and install requests if needed
requests_available = False
if not check_module("requests"):
    print("Mocai: 'requests' module not found. Attempting automatic installation...")
    if ensure_pip():
        if install_module("requests"):
            requests_available = check_module("requests")
        else:
            print("Mocai: ERROR - Could not install 'requests'. Please install manually:")
            print(f"Mocai: Run: {sys.executable} -m pip install requests")
    else:
        print("Mocai: ERROR - pip not available. Please install 'requests' manually.")
else:
    requests_available = True

# Import Blender API
import bpy

if requests_available:
    # All dependencies available, proceed with normal registration
    try:
        from . import properties
        from . import operators
        from . import ui

        def register():
            """Register add-on"""
            properties.register()
            operators.register()
            ui.register()
            print("Mocai add-on registered successfully")

        def unregister():
            """Unregister add-on"""
            ui.unregister()
            operators.unregister()
            properties.unregister()
            print("Mocai add-on unregistered")

    except ImportError as e:
        print(f"Mocai: Error importing add-on modules: {e}")

        def register():
            print("Mocai: Add-on registration failed due to import error")

        def unregister():
            pass

else:
    # Dependencies not available, register with error message
    class MOCAI_PT_error_panel(bpy.types.Panel):
        """Error panel when dependencies are missing"""
        bl_label = "Mocai - Error"
        bl_idname = "MOCAI_PT_error_panel"
        bl_space_type = 'VIEW_3D'
        bl_region_type = 'UI'
        bl_category = 'Mocai'

        def draw(self, context):
            layout = self.layout
            box = layout.box()
            box.label(text="Missing Dependencies", icon='ERROR')
            box.label(text="The 'requests' module could not be installed.")
            box.label(text="")
            box.label(text="Please install manually:")
            box.label(text=f"{sys.executable}")
            box.label(text="-m pip install requests")

    def register():
        """Register error panel"""
        bpy.utils.register_class(MOCAI_PT_error_panel)
        print("Mocai: Add-on loaded with errors - missing dependencies")

    def unregister():
        """Unregister error panel"""
        bpy.utils.unregister_class(MOCAI_PT_error_panel)


if __name__ == "__main__":
    register()
